<?php
namespace DDTrace\Contracts {
use DDTrace\Exceptions\InvalidReferencesSet;
use DDTrace\Exceptions\InvalidSpanOption;
use DDTrace\Exceptions\UnsupportedFormat;
use DDTrace\StartSpanOptions;
interface Tracer
{
    public function limited();
    public function getScopeManager();
    public function getActiveSpan();
    public function startActiveSpan($operationName, $options = []);
    public function startSpan($operationName, $options = []);
    public function inject(SpanContext $spanContext, $format, &$carrier);
    public function extract($format, $carrier);
    public function flush();
    public function setPrioritySampling($prioritySampling);
    public function getPrioritySampling();
    public function startRootSpan($operationName, $options = []);
    public function getRootScope();
    public function getSafeRootSpan();
    public function getTracesAsArray();
    public function getTracesCount();
}
}
namespace DDTrace\Contracts {
interface Span
{
    public function getOperationName();
    public function getContext();
    public function finish($finishTime = null);
    public function overwriteOperationName($newOperationName);
    public function setResource($resource);
    public function setTag($key, $value, $setIfFinished = false);
    public function getTag($key);
    public function log(array $fields = [], $timestamp = null);
    public function addBaggageItem($key, $value);
    public function getBaggageItem($key);
    public function getAllBaggageItems();
    public function setError($error);
    public function setRawError($message, $type);
    public function hasError();
    public function getStartTime();
    public function getDuration();
    public function getTraceId();
    public function getSpanId();
    public function getParentId();
    public function getResource();
    public function getService();
    public function getType();
    public function isFinished();
    public function getAllTags();
    public function hasTag($name);
    public function setMetric($key, $value);
    public function getMetrics();
}
}
namespace DDTrace\Contracts {
interface Scope
{
    public function close();
    public function getSpan();
}
}
namespace DDTrace\Contracts {
interface ScopeManager
{
    const DEFAULT_FINISH_SPAN_ON_CLOSE = true;
    public function activate(Span $span, $finishSpanOnClose = self::DEFAULT_FINISH_SPAN_ON_CLOSE);
    public function getActive();
    public function close();
}
}
namespace DDTrace\Contracts {
use IteratorAggregate;
interface SpanContext extends IteratorAggregate
{
    public function getBaggageItem($key);
    public function withBaggageItem($key, $value);
    public function getAllBaggageItems();
    public function getPropagatedPrioritySampling();
    public function setPropagatedPrioritySampling($propagatedPrioritySampling);
    public function isHostRoot();
    public function getTraceId();
    public function getSpanId();
    public function getParentId();
    public function isDistributedTracingActivationContext();
}
}
namespace DDTrace\Data {
use DDTrace\Contracts\SpanContext as SpanContextInterface;
abstract class SpanContext implements SpanContextInterface
{
    public $traceId;
    public $spanId;
    public $parentId;
    public $isDistributedTracingActivationContext;
    public $propagatedPrioritySampling;
    public $origin;
    public $parentContext;
    public $baggageItems;
}
}
namespace DDTrace\Data {
use DDTrace\Data\SpanContext as SpanContextData;
use DDTrace\Contracts\Span as SpanInterface;
abstract class Span implements SpanInterface
{
    public $context;
    public $hasError = false;
    protected $internalSpan;
    public function &__get($name)
    {
        if ($name == "operationName") {
            $name = "name";
        } elseif ($name == "tags") {
            $name = "meta";
        } elseif ($name == "duration") {
            $duration = $this->internalSpan->getDuration();
            return $duration;
        } elseif ($name == "startTime") {
            $startTime = $this->internalSpan->getStartTime();
            return $startTime;
        }
        return $this->internalSpan->{$name};
    }
    public function __set($name, $value)
    {
        if ($name == "operationName") {
            $name = "name";
        } elseif ($name == "tags") {
            $name = "meta";
        }
        return $this->internalSpan->{$name} = $value;
    }
    public function __isset($name)
    {
        return $this->__get($name) !== null;
    }
}
}
namespace DDTrace\Sampling {
class PrioritySampling
{
    const USER_REJECT = -1;
    const AUTO_REJECT = 0;
    const AUTO_KEEP = 1;
    const USER_KEEP = 2;
    const UNKNOWN = null;
    public static function parse($value)
    {
        if (!is_numeric($value)) {
            return self::UNKNOWN;
        }
        $intValue = intval($value);
        return in_array($intValue, [self::USER_REJECT, self::AUTO_KEEP, self::AUTO_REJECT, self::USER_KEEP]) ? $intValue : self::UNKNOWN;
    }
}
}
namespace DDTrace {
use DDTrace\Contracts\Tracer as TracerInterface;
interface Transport
{
    public function send(TracerInterface $tracer);
    public function setHeader($key, $value);
}
}
namespace DDTrace\Obfuscation {
final class WildcardToRegex
{
    const REPLACEMENT_CHARACTER = '?';
    public static function convert($wildcardPattern)
    {
        $wildcardPattern = $replacement = trim($wildcardPattern);
        if (false !== strpos($replacement, '$*')) {
            $replacement = str_replace('%', '%%', $replacement);
            $replacementCount = 0;
            $replacement = str_replace('$*', '%s', $replacement, $replacementCount);
            $sprintfArgs = [$replacement];
            for ($i = 1; $i <= $replacementCount; $i++) {
                $sprintfArgs[] = '${' . $i . '}';
            }
            $replacement = call_user_func_array('sprintf', $sprintfArgs);
        }
        $replacement = str_replace('*', self::REPLACEMENT_CHARACTER, $replacement);
        return ['|^' . str_replace(['\$\*', '\*'], ['(.+)', '.+'], preg_quote($wildcardPattern, '|')) . '$|', $replacement];
    }
}
}
namespace DDTrace {
use DDTrace\Contracts\Span as SpanInterface;
use DDTrace\Contracts\SpanContext as SpanContextInterface;
use DDTrace\Exceptions\InvalidReferenceArgument;
final class Reference
{
    const CHILD_OF = 'child_of';
    const FOLLOWS_FROM = 'follows_from';
    private $type;
    private $context;
    private function __construct($type, SpanContextInterface $context)
    {
        $this->type = $type;
        $this->context = $context;
    }
    public static function create($type, $context)
    {
        if (empty($type)) {
            throw InvalidReferenceArgument::forEmptyType();
        }
        return new self($type, self::extractContext($context));
    }
    public function getContext()
    {
        return $this->context;
    }
    public function isType($type)
    {
        return $this->type === $type;
    }
    private static function extractContext($context)
    {
        if ($context instanceof SpanContextInterface) {
            return $context;
        }
        if ($context instanceof SpanInterface) {
            return $context->getContext();
        }
        throw InvalidReferenceArgument::forInvalidContext($context);
    }
}
}
namespace DDTrace\Exceptions {
use InvalidArgumentException;
final class InvalidReferenceArgument extends InvalidArgumentException
{
    public static function forEmptyType()
    {
        return new self('Reference type can not be an empty string');
    }
    public static function forInvalidContext($context)
    {
        return new self(sprintf('Reference expects \DDTrace\Contracts\Span or \DDTrace\Contracts\SpanContext as context, got %s', is_object($context) ? get_class($context) : gettype($context)));
    }
}
}
namespace DDTrace\Exceptions {
use UnexpectedValueException;
final class UnsupportedFormat extends UnexpectedValueException
{
    public static function forFormat($format)
    {
        return new self(sprintf('The format \'%s\' is not supported.', $format));
    }
}
}
namespace DDTrace\Exceptions {
use InvalidArgumentException;
final class InvalidSpanArgument extends InvalidArgumentException
{
    public static function forTagKey($key)
    {
        return new self(sprintf('Invalid key type in given span tags. Expected string, got %s.', gettype($key)));
    }
    public static function forError($error)
    {
        return new self(sprintf('Error should be either Exception or Throwable, got %s.', gettype($error)));
    }
}
}
namespace DDTrace\Exceptions {
use DomainException;
final class InvalidReferencesSet extends DomainException
{
    public static function create($message)
    {
        return new self($message);
    }
    public static function forMoreThanOneParent()
    {
        return new self('Span can not have more than one parent, either one as child_of or either one as follows_from');
    }
}
}
namespace DDTrace\Exceptions {
use InvalidArgumentException;
final class InvalidSpanOption extends InvalidArgumentException
{
    public static function forIncludingBothChildOfAndReferences()
    {
        return new self('Either "childOf" or "references" options are accepted but not both.');
    }
    public static function forInvalidReference($reference)
    {
        return new self(sprintf('Invalid reference. Expected \DDTrace\Reference, got %s.', is_object($reference) ? get_class($reference) : gettype($reference)));
    }
    public static function forInvalidStartTime()
    {
        return new self(sprintf('Invalid start_time option. Expected int or float got string.'));
    }
    public static function forInvalidChildOf($childOfOption)
    {
        return new self(sprintf('Invalid child_of option. Expected Span or SpanContext, got %s', is_object($childOfOption) ? get_class($childOfOption) : gettype($childOfOption)));
    }
    public static function forUnknownOption($key)
    {
        return new self(sprintf('Invalid option %s.', $key));
    }
    public static function forInvalidTag($tag)
    {
        return new self(sprintf('Invalid tag. Expected string, got %s', gettype($tag)));
    }
    public static function forInvalidTagValue($tagValue)
    {
        return new self(sprintf('Invalid tag value. Expected scalar or object with __toString method, got %s', is_object($tagValue) ? get_class($tagValue) : gettype($tagValue)));
    }
    public static function forInvalidTags($value)
    {
        return new self(sprintf('Invalid tags value. Expected a associative array of tags, got %s', is_object($value) ? get_class($value) : gettype($value)));
    }
    public static function forInvalidReferenceSet($value)
    {
        return new self(sprintf('Invalid references set. Expected Reference or Reference[], got %s', is_object($value) ? get_class($value) : gettype($value)));
    }
    public static function forFinishSpanOnClose($value)
    {
        return new self(sprintf('Invalid type for finish_span_on_close. Expected bool, got %s', is_object($value) ? get_class($value) : gettype($value)));
    }
    public static function forIgnoreActiveSpan($value)
    {
        return new self(sprintf('Invalid type for ignore_active_span. Expected bool, got %s', is_object($value) ? get_class($value) : gettype($value)));
    }
}
}
namespace DDTrace {
use DDTrace\Contracts\Tracer as TracerInterface;
final class GlobalTracer
{
    private static $instance;
    public static function set(TracerInterface $tracer)
    {
        self::$instance = $tracer;
    }
    public static function get()
    {
        if (null !== self::$instance) {
            return self::$instance;
        }
        if (\extension_loaded('ddtrace') && class_exists(Tracer::class)) {
            return self::$instance = new Tracer();
        }
        return self::$instance = NoopTracer::create();
    }
}
}
namespace DDTrace {
class Format
{
    const BINARY = 'binary';
    const TEXT_MAP = 'text_map';
    const HTTP_HEADERS = 'http_headers';
}
}
namespace DDTrace\Log {
interface LoggerInterface
{
    public function debug($message, array $context = array());
    public function warning($message, array $context = []);
    public function error($message, array $context = array());
    public function isLevelActive($level);
}
}
namespace DDTrace\Log {
trait InterpolateTrait
{
    public static function interpolate($message, array $context = [])
    {
        $replace = array();
        foreach ($context as $key => $val) {
            if (is_array($val)) {
                $val = "{Array of size " . count($val) . "}";
            } elseif (is_object($val) && !method_exists($val, '__toString')) {
                $val = "{Object of type " . get_class($val) . "}";
            }
            $replace['{' . $key . '}'] = $val;
        }
        return strtr($message, $replace);
    }
}
}
namespace DDTrace\Log {
final class LogLevel
{
    const EMERGENCY = 'emergency';
    const ALERT = 'alert';
    const CRITICAL = 'critical';
    const ERROR = 'error';
    const WARNING = 'warning';
    const NOTICE = 'notice';
    const INFO = 'info';
    const DEBUG = 'debug';
    public static function all()
    {
        return [self::EMERGENCY, self::ALERT, self::CRITICAL, self::ERROR, self::WARNING, self::NOTICE, self::INFO, self::DEBUG];
    }
}
}
namespace DDTrace\Log {
final class Logger
{
    private static $logger;
    public static function set(LoggerInterface $logger)
    {
        self::$logger = $logger;
    }
    public static function get()
    {
        if (self::$logger === null) {
            self::$logger = \function_exists("dd_trace_env_config") && \dd_trace_env_config("DD_TRACE_DEBUG") ? new ErrorLogLogger(LogLevel::DEBUG) : new NullLogger(LogLevel::EMERGENCY);
        }
        return self::$logger;
    }
    public static function reset()
    {
        self::$logger = null;
    }
}
}
namespace DDTrace\Log {
abstract class AbstractLogger implements LoggerInterface
{
    private $enabledLevels = [];
    public function __construct($level)
    {
        $level = trim(strtolower($level));
        # all() have all levels ordered from highest to lowest
        # all preceding levels to given $level will be marked as enabled
        $enabled = false;
        foreach (array_reverse(LogLevel::all()) as $knownLevel) {
            $enabled = $enabled || $level === $knownLevel;
            $this->enabledLevels[$knownLevel] = $enabled;
        }
    }
    public function isLevelActive($level)
    {
        return (bool) $this->enabledLevels[$level];
    }
}
}
namespace DDTrace\Log {
final class DatadogLogger
{
    use InterpolateTrait;
    const DEFAULT_JSON_FLAGS = JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE | JSON_PRESERVE_ZERO_FRACTION | JSON_PARTIAL_OUTPUT_ON_ERROR;
    protected $stream;
    private $errorMessage = null;
    private $dirCreated = null;
    public function __construct($stream = null, $mode = 'a')
    {
        if (is_null($stream)) {
            $stream = (\dd_trace_env_config('DD_TRACE_LOG_FILE') ?: ini_get('error_log')) ?: 'php://stderr';
            if ($stream === '/dev/stderr') {
                $stream = 'php://stderr';
            } elseif ($stream === '/dev/stdout') {
                $stream = 'php://stdout';
            }
        }
        if (is_resource($stream)) {
            $this->stream = $stream;
        } elseif (is_string($stream)) {
            $url = self::canonicalizePath($stream);
            $this->createDir($url);
            $this->createStream($url, $mode);
        }
    }
    public function emergency($message, array $context = [])
    {
        $this->log(LogLevel::EMERGENCY, $message, $context);
    }
    public function alert($message, array $context = [])
    {
        $this->log(LogLevel::ALERT, $message, $context);
    }
    public function critical($message, array $context = [])
    {
        $this->log(LogLevel::CRITICAL, $message, $context);
    }
    public function error($message, array $context = [])
    {
        $this->log(LogLevel::ERROR, $message, $context);
    }
    public function warning($message, array $context = [])
    {
        $this->log(LogLevel::WARNING, $message, $context);
    }
    public function notice($message, array $context = [])
    {
        $this->log(LogLevel::NOTICE, $message, $context);
    }
    public function info($message, array $context = [])
    {
        $this->log(LogLevel::INFO, $message, $context);
    }
    public function debug($message, array $context = [])
    {
        $this->log(LogLevel::DEBUG, $message, $context);
    }
    public function log(string $level, $message, array $context = [])
    {
        if (self::isLogLevelValid($level)) {
            $this->emit(self::format($level, $message, $context));
        }
    }
    private static function isLogLevelValid(string $level): bool
    {
        return \in_array($level, LogLevel::all());
    }
    private static function format(string $level, $message, array $context = []): string
    {
        $message = self::interpolate($message, $context);
        $date = \DateTime::createFromFormat('U.u', microtime(true));
        $record = ['message' => $message, 'status' => $level, 'timestamp' => $date->format('Y-m-d\TH:i:s.uP')] + $context;
        return json_encode(array_merge($record, self::handleLogInjection()), self::DEFAULT_JSON_FLAGS) . PHP_EOL;
    }
    private static function handleLogInjection(): array
    {
        $logInjection = \dd_trace_env_config('DD_LOGS_INJECTION');
        if ($logInjection) {
            $traceId = \DDTrace\logs_correlation_trace_id();
            $spanId = \dd_trace_peek_span_id();
            if ($traceId && $spanId) {
                return ['dd.trace_id' => $traceId, 'dd.span_id' => $spanId];
            }
        }
        return [];
    }
    private function emit(string $message)
    {
        if ($this->stream) {
            fwrite($this->stream, $message);
        }
    }
    private static function canonicalizePath(string $streamUrl): string
    {
        $prefix = '';
        if ('file://' === substr($streamUrl, 0, 7)) {
            $streamUrl = substr($streamUrl, 7);
            $prefix = 'file://';
        }
        if (false !== strpos($streamUrl, '://')) {
            return $streamUrl;
        }
        if (substr($streamUrl, 0, 1) === '/' || substr($streamUrl, 1, 1) === ':' || substr($streamUrl, 0, 2) === '\\\\') {
            return $prefix . $streamUrl;
        }
        $streamUrl = getcwd() . '/' . $streamUrl;
        return $prefix . $streamUrl;
    }
    private function createDir(string $url)
    {
        if ($this->dirCreated) {
            return;
        }
        $dir = self::getDirFromStream($url);
        if (null !== $dir && !is_dir($dir)) {
            $this->errorMessage = null;
            set_error_handler([$this, 'customErrorHandler']);
            $status = mkdir($dir, 0666, true);
            restore_error_handler();
            if (false === $status && !is_dir($dir) && strpos((string) $this->errorMessage, 'File exists') === false) {
                return;
            }
        }
        $this->dirCreated = true;
    }
    private function createStream(string $url, string $mode)
    {
        $this->errorMessage = null;
        set_error_handler([$this, 'customErrorHandler']);
        $this->stream = fopen($url, $mode);
        restore_error_handler();
    }
    public function customErrorHandler(int $code, string $msg): bool
    {
        $this->errorMessage = preg_replace('{^(fopen|mkdir)\(.*?\): }', '', $msg);
        return true;
    }
    private static function getDirFromStream(string $stream)
    {
        $pos = strpos($stream, '://');
        if ($pos === false) {
            return dirname($stream);
        }
        if ('file://' === substr($stream, 0, 7)) {
            return dirname(substr($stream, 7));
        }
        return null;
    }
}
}
namespace DDTrace\Log {
class ErrorLogLogger extends AbstractLogger
{
    use InterpolateTrait;
    public function debug($message, array $context = [])
    {
        $this->emit(LogLevel::DEBUG, $message, $context);
    }
    public function warning($message, array $context = [])
    {
        $this->emit(LogLevel::WARNING, $message, $context);
    }
    public function error($message, array $context = [])
    {
        $this->emit(LogLevel::ERROR, $message, $context);
    }
    private function emit($level, $message, array $context = [])
    {
        if (!$this->isLevelActive($level)) {
            return;
        }
        $interpolatedMessage = $this->interpolate($message, $context);
        $date = date(\DateTime::ATOM);
        error_log("[{$date}] [ddtrace] [{$level}] - {$interpolatedMessage}");
    }
}
}
namespace DDTrace\Log {
final class NullLogger extends AbstractLogger
{
    public function debug($message, array $context = array())
    {
    }
    public function warning($message, array $context = [])
    {
    }
    public function error($message, array $context = array())
    {
    }
    public function isLevelActive($level)
    {
        return false;
    }
}
}
namespace DDTrace\Log {
trait LoggingTrait
{
    protected static function logDebug($message, array $context = [])
    {
        Logger::get()->debug($message, $context);
    }
    protected static function logWarning($message, array $context = [])
    {
        Logger::get()->warning($message, $context);
    }
    protected static function logError($message, array $context = [])
    {
        Logger::get()->error($message, $context);
    }
    protected static function isLogDebugActive()
    {
        return Logger::get()->isLevelActive(LogLevel::DEBUG);
    }
}
}
namespace DDTrace {
use DDTrace\Contracts\ScopeManager as ScopeManagerInterface;
use DDTrace\Contracts\Span as SpanInterface;
use DDTrace\Contracts\SpanContext as SpanContextInterface;
use DDTrace\Exceptions\InvalidReferencesSet;
use DDTrace\Exceptions\InvalidSpanOption;
final class StartSpanOptions
{
    private $references = [];
    private $tags = [];
    private $startTime;
    private $finishSpanOnClose = ScopeManagerInterface::DEFAULT_FINISH_SPAN_ON_CLOSE;
    private $ignoreActiveSpan = false;
    public static function create(array $options)
    {
        $spanOptions = new self();
        foreach ($options as $key => $value) {
            switch ($key) {
                case 'child_of':
                    if (!empty($spanOptions->references)) {
                        throw InvalidSpanOption::forIncludingBothChildOfAndReferences();
                    }
                    $spanOptions->references[] = self::buildChildOf($value);
                    break;
                case 'references':
                    if (!empty($spanOptions->references)) {
                        throw InvalidSpanOption::forIncludingBothChildOfAndReferences();
                    }
                    if ($value instanceof Reference) {
                        $spanOptions->references = [$value];
                    } elseif (is_array($value)) {
                        $spanOptions->references = self::buildReferences($value);
                    } else {
                        throw InvalidSpanOption::forInvalidReferenceSet($value);
                    }
                    break;
                case 'tags':
                    if (!is_array($value)) {
                        throw InvalidSpanOption::forInvalidTags($value);
                    }
                    foreach ($value as $tag => $tagValue) {
                        if ($tag !== (string) $tag) {
                            throw InvalidSpanOption::forInvalidTag($tag);
                        }
                        $spanOptions->tags[$tag] = $tagValue;
                    }
                    break;
                case 'start_time':
                    if (is_scalar($value) && !is_numeric($value)) {
                        throw InvalidSpanOption::forInvalidStartTime();
                    }
                    $spanOptions->startTime = $value;
                    break;
                case 'finish_span_on_close':
                    if (!is_bool($value)) {
                        throw InvalidSpanOption::forFinishSpanOnClose($value);
                    }
                    $spanOptions->finishSpanOnClose = $value;
                    break;
                case 'ignore_active_span':
                    if (!is_bool($value)) {
                        throw InvalidSpanOption::forIgnoreActiveSpan($value);
                    }
                    $spanOptions->ignoreActiveSpan = $value;
                    break;
                default:
                    throw InvalidSpanOption::forUnknownOption($key);
                    break;
            }
        }
        return $spanOptions;
    }
    public function withParent($parent)
    {
        $newSpanOptions = new StartSpanOptions();
        $newSpanOptions->references[] = self::buildChildOf($parent);
        $newSpanOptions->tags = $this->tags;
        $newSpanOptions->startTime = $this->startTime;
        $newSpanOptions->finishSpanOnClose = $this->finishSpanOnClose;
        $newSpanOptions->ignoreActiveSpan = $this->ignoreActiveSpan;
        return $newSpanOptions;
    }
    public function getReferences()
    {
        return $this->references;
    }
    public function getTags()
    {
        return $this->tags;
    }
    public function getStartTime()
    {
        return $this->startTime;
    }
    public function shouldFinishSpanOnClose()
    {
        return $this->finishSpanOnClose;
    }
    public function shouldIgnoreActiveSpan()
    {
        return $this->ignoreActiveSpan;
    }
    private static function buildChildOf($value)
    {
        if ($value instanceof SpanInterface) {
            return Reference::create(Reference::CHILD_OF, $value->getContext());
        }
        if ($value instanceof SpanContextInterface) {
            return Reference::create(Reference::CHILD_OF, $value);
        }
        if ($value instanceof \DDTrace\OpenTracer\SpanContext || $value instanceof \DDTrace\OpenTracer1\SpanContext) {
            return Reference::create(Reference::CHILD_OF, $value->unwrapped());
        }
        throw InvalidSpanOption::forInvalidChildOf($value);
    }
    private static function buildReferences(array $referencesArray)
    {
        $references = [];
        foreach ($referencesArray as $reference) {
            if (!$reference instanceof Reference) {
                throw InvalidSpanOption::forInvalidReference($reference);
            }
            $references[] = $reference;
        }
        return $references;
    }
}
}
namespace DDTrace {
use ArrayIterator;
use DDTrace\Contracts\SpanContext as SpanContextInterface;
use DDTrace\Data\SpanContext as SpanContextData;
final class SpanContext extends SpanContextData
{
    public function __construct($traceId, $spanId, $parentId = null, array $baggageItems = [], $isDistributedTracingActivationContext = false)
    {
        $this->traceId = $traceId;
        $this->spanId = $spanId;
        $this->parentId = $parentId ?: null;
        $this->baggageItems = $baggageItems;
        $this->isDistributedTracingActivationContext = $isDistributedTracingActivationContext;
    }
    public static function createAsChild(SpanContextInterface $parentContext, $startTime = null)
    {
        $origin = \property_exists($parentContext, 'origin') ? $parentContext->origin : null;
        if (!$parentContext->isDistributedTracingActivationContext() || !active_span()) {
            if ($parentContext->isDistributedTracingActivationContext()) {
                set_distributed_tracing_context($parentContext->getTraceId(), $parentContext->getSpanId(), $origin);
            }
            if ($startTime) {
                start_span($startTime);
            } else {
                start_span();
            }
        }
        $instance = new self($parentContext->getTraceId(), active_span()->id, $parentContext->getSpanId(), $parentContext->getAllBaggageItems(), false);
        $instance->parentContext = $parentContext;
        $instance->setPropagatedPrioritySampling($parentContext->getPropagatedPrioritySampling());
        if ($origin) {
            $instance->origin = $origin;
        }
        return $instance;
    }
    public static function createAsRoot(array $baggageItems = [], $startTime = null)
    {
        if (!active_span()) {
            if ($startTime) {
                start_span($startTime);
            } else {
                start_span();
            }
        }
        $nextId = active_span()->id;
        return new self($nextId, $nextId, null, $baggageItems, false);
    }
    public function getTraceId()
    {
        return $this->traceId;
    }
    public function getSpanId()
    {
        return $this->spanId;
    }
    public function getParentId()
    {
        return $this->parentId;
    }
    #[\ReturnTypeWillChange]
    public function getIterator()
    {
        return new ArrayIterator($this->baggageItems);
    }
    public function getPropagatedPrioritySampling()
    {
        return $this->propagatedPrioritySampling;
    }
    public function setPropagatedPrioritySampling($propagatedPrioritySampling)
    {
        $this->propagatedPrioritySampling = $propagatedPrioritySampling;
    }
    public function getBaggageItem($key)
    {
        return array_key_exists($key, $this->baggageItems) ? $this->baggageItems[$key] : null;
    }
    public function withBaggageItem($key, $value)
    {
        return new self($this->traceId, $this->spanId, $this->parentId, array_merge($this->baggageItems, [$key => $value]));
    }
    public function getAllBaggageItems()
    {
        return $this->baggageItems;
    }
    public function isEqual(SpanContextInterface $spanContext)
    {
        if ($spanContext instanceof SpanContextData) {
            return $this->traceId === $spanContext->traceId && $this->spanId === $spanContext->spanId && $this->parentId === $spanContext->parentId && $this->baggageItems === $spanContext->baggageItems;
        }
        return $this->traceId === $spanContext->getTraceId() && $this->spanId === $spanContext->getSpanId() && $this->parentId === $spanContext->getParentId() && $this->baggageItems === $spanContext->getAllBaggageItems();
    }
    public function isDistributedTracingActivationContext()
    {
        return $this->isDistributedTracingActivationContext;
    }
    public function isHostRoot()
    {
        return $this->parentContext === null || $this->parentContext->isDistributedTracingActivationContext();
    }
}
}
namespace DDTrace {
use DDTrace\Data\Span as DataSpan;
use DDTrace\Exceptions\InvalidSpanArgument;
use DDTrace\Log\LoggingTrait;
use DDTrace\SpanContext as SpanContext;
use DDTrace\Http\Urls;
use DDTrace\Processing\TraceAnalyticsProcessor;
use Exception;
use InvalidArgumentException;
use Throwable;
class Span extends DataSpan
{
    use LoggingTrait;
    private static $metricNames = [Tag::ANALYTICS_KEY => true];
    private static $specialTags = [Tag::ANALYTICS_KEY => true, Tag::ERROR => true, Tag::ERROR_MSG => true, Tag::SERVICE_NAME => true, Tag::RESOURCE_NAME => true, Tag::SPAN_TYPE => true, Tag::HTTP_URL => true, Tag::HTTP_STATUS_CODE => true, Tag::MANUAL_KEEP => true, Tag::MANUAL_DROP => true, Tag::SERVICE_VERSION => true];
    public $internalSpan;
    public function __construct(SpanData $internalSpan, SpanContext $context)
    {
        $this->internalSpan = $internalSpan;
        $this->context = $context;
        $internalSpan->baggage += $context->baggageItems;
        $context->baggageItems =& $internalSpan->baggage;
    }
    public function getTraceId()
    {
        return $this->context->traceId;
    }
    public function getSpanId()
    {
        return $this->context->spanId;
    }
    public function getParentId()
    {
        return $this->context->parentId;
    }
    public function overwriteOperationName($operationName)
    {
        $this->internalSpan->name = $operationName;
    }
    public function getResource()
    {
        return $this->internalSpan->resource;
    }
    public function getService()
    {
        return $this->internalSpan->service;
    }
    public function getType()
    {
        return $this->internalSpan->type;
    }
    public function getStartTime()
    {
        return (int) ($this->startTime / 1000);
    }
    public function getDuration()
    {
        return (int) ($this->duration / 1000);
    }
    public function setTag($key, $value, $setIfFinished = false)
    {
        if ($this->isFinished() && !$setIfFinished) {
            return;
        }
        if ($key !== (string) $key) {
            throw InvalidSpanArgument::forTagKey($key);
        }
        if (is_object($value) && $key !== Tag::ERROR) {
            return;
        }
        if (array_key_exists($key, self::$specialTags)) {
            if ($key === Tag::ERROR) {
                $this->setError($value);
                return;
            }
            if ($key === Tag::ERROR_MSG) {
                $this->internalSpan->meta[$key] = (string) $value;
                $this->setError(true);
                return;
            }
            if ($key === Tag::SERVICE_NAME) {
                $this->internalSpan->service = $value;
                return;
            }
            if ($key === Tag::MANUAL_KEEP) {
                set_priority_sampling(Sampling\PrioritySampling::USER_KEEP);
                return;
            }
            if ($key === Tag::MANUAL_DROP) {
                set_priority_sampling(Sampling\PrioritySampling::USER_REJECT);
                return;
            }
            if ($key === Tag::RESOURCE_NAME) {
                $this->internalSpan->resource = (string) $value;
                return;
            }
            if ($key === Tag::SPAN_TYPE) {
                $this->internalSpan->type = $value;
                return;
            }
            if ($key === Tag::HTTP_URL) {
                $value = Urls::sanitize((string) $value);
            }
            if ($key === Tag::HTTP_STATUS_CODE && $value >= 500) {
                $this->hasError = true;
                if (!isset($this->internalSpan->meta[Tag::ERROR_TYPE])) {
                    $this->internalSpan->meta[Tag::ERROR_TYPE] = 'Internal Server Error';
                }
            }
            if ($key === Tag::SERVICE_VERSION) {
                $this->setTag(Tag::VERSION, $value);
            }
            if (array_key_exists($key, self::$metricNames)) {
                $this->setMetric($key, $value);
                return;
            }
        }
        $this->internalSpan->meta[$key] = (string) $value;
    }
    public function getTag($key)
    {
        if (isset($this->internalSpan->meta) && array_key_exists($key, $this->internalSpan->meta)) {
            return $this->internalSpan->meta[$key];
        } elseif ($key === "env") {
            return $this->internalSpan->env;
        } elseif ($key === "version") {
            return $this->internalSpan->version;
        }
        return null;
    }
    public function getAllTags()
    {
        return isset($this->internalSpan->meta) ? $this->internalSpan->meta : [];
    }
    public function hasTag($name)
    {
        return array_key_exists($name, $this->getAllTags());
    }
    public function setMetric($key, $value)
    {
        if ($key === Tag::ANALYTICS_KEY) {
            TraceAnalyticsProcessor::normalizeAnalyticsValue($this->internalSpan->metrics, $value);
            return;
        }
        $this->internalSpan->metrics[$key] = $value;
    }
    public function getMetrics()
    {
        return isset($this->internalSpan->metrics) ? $this->internalSpan->metrics : [];
    }
    public function setResource($resource)
    {
        $this->internalSpan->resource = (string) $resource;
    }
    public function setError($error)
    {
        if ($error instanceof Exception || $error instanceof Throwable) {
            $this->hasError = true;
            $this->internalSpan->meta[Tag::ERROR_MSG] = $error->getMessage();
            $this->internalSpan->meta[Tag::ERROR_TYPE] = get_class($error);
            $this->internalSpan->meta[Tag::ERROR_STACK] = $error->getTraceAsString();
            return;
        }
        if (is_bool($error)) {
            $this->hasError = $error;
            return;
        }
        if (is_null($error)) {
            $this->hasError = false;
        }
        throw InvalidSpanArgument::forError($error);
    }
    public function setRawError($message, $type)
    {
        $this->hasError = true;
        $this->internalSpan->meta[Tag::ERROR_MSG] = $message;
        $this->internalSpan->meta[Tag::ERROR_TYPE] = $type;
    }
    public function hasError()
    {
        return $this->hasError;
    }
    public function finish($finishTime = null)
    {
        if (!$this->isFinished()) {
            $current_span = active_span();
            if ($current_span != $this->internalSpan) {
                if (close_spans_until($this->internalSpan)) {
                    self::logWarning("Closed multiple spans at once while finishing a span named '{name}'. " . "The topmost span was named '{topname}'.", ["name" => $this->getOperationName(), "topname" => $current_span->name]);
                } else {
                    self::logDebug("Found unexpected span with name '{name}' on top of the span stack.", ["name" => $current_span->name]);
                }
            }
            close_span($finishTime ?: 0);
        }
    }
    public function isFinished()
    {
        return $this->duration !== 0;
    }
    public function getOperationName()
    {
        return $this->internalSpan->name;
    }
    public function getContext()
    {
        return $this->context;
    }
    public function log(array $fields = [], $timestamp = null)
    {
        foreach ($fields as $key => $value) {
            if ($key === Tag::LOG_EVENT && $value === Tag::ERROR) {
                $this->setError(true);
            } elseif ($key === Tag::LOG_ERROR || $key === Tag::LOG_ERROR_OBJECT) {
                $this->setError($value);
            } elseif ($key === Tag::LOG_MESSAGE) {
                $this->internalSpan->meta[Tag::ERROR_MSG] = (string) $value;
            } elseif ($key === Tag::LOG_STACK) {
                $this->setTag(Tag::ERROR_STACK, $value);
            }
        }
    }
    public function addBaggageItem($key, $value)
    {
        if ($key !== '') {
            $this->internalSpan->baggage[$key] = $value;
        }
    }
    public function getBaggageItem($key)
    {
        return $this->internalSpan->baggage[$key] ?? null;
    }
    public function getAllBaggageItems()
    {
        return $this->internalSpan->baggage;
    }
    public function removeBaggageItem($key)
    {
        unset($this->internalSpan->baggage[$key]);
    }
    public function removeAllBaggageItems()
    {
        $this->internalSpan->baggage = [];
    }
    public function __destruct()
    {
        if (!$this->isFinished()) {
            self::logWarning("The span with name '{name}' was closed without being finished first.", ["name" => $this->getOperationName()]);
        }
    }
}
}
namespace DDTrace {
class Time
{
    public static function now()
    {
        return (int) (microtime(true) * 1000 * 1000);
    }
    public static function fromMicrotime($microtime)
    {
        return (int) $microtime * 1000 * 1000;
    }
    public static function isValid($time)
    {
        return $time === (int) $time && ctype_digit((string) $time) && strlen((string) $time) === 16;
    }
}
}
namespace DDTrace\Transport {
use DDTrace\Contracts\Tracer as TracerInterface;
use DDTrace\Transport;
final class Internal implements Transport
{
    public function send(TracerInterface $tracer)
    {
        if (!\dd_trace_env_config('DD_TRACE_GENERATE_ROOT_SPAN')) {
            \DDTrace\flush();
        }
    }
    public function setHeader($key, $value)
    {
    }
}
}
namespace DDTrace\Processing {
use DDTrace\Data\Span as DataSpan;
use DDTrace\Tag;
final class TraceAnalyticsProcessor
{
    public static function normalizeAnalyticsValue(&$metrics, $value)
    {
        if (true === $value) {
            $metrics[Tag::ANALYTICS_KEY] = 1.0;
        } elseif (false === $value) {
            unset($metrics[Tag::ANALYTICS_KEY]);
        } elseif (is_numeric($value) && 0 <= $value && $value <= 1) {
            $metrics[Tag::ANALYTICS_KEY] = (float) $value;
        }
    }
}
}
namespace DDTrace {
use DDTrace\Contracts\Scope as ScopeInterface;
use DDTrace\Contracts\Span as SpanInterface;
final class Scope implements ScopeInterface
{
    private $span;
    private $scopeManager;
    private $finishSpanOnClose;
    public function __construct(ScopeManager $scopeManager, SpanInterface $span, $finishSpanOnClose)
    {
        $this->scopeManager = $scopeManager;
        $this->span = $span;
        $this->finishSpanOnClose = $finishSpanOnClose;
    }
    public function close()
    {
        if ($this->finishSpanOnClose) {
            $this->span->finish();
        }
        $this->scopeManager->deactivate($this);
    }
    public function getSpan()
    {
        return $this->span;
    }
}
}
namespace DDTrace {
use DDTrace\Contracts\SpanContext as SpanContextInterface;
interface Propagator
{
    const DEFAULT_BAGGAGE_HEADER_PREFIX = 'ot-baggage-';
    const DEFAULT_TRACE_ID_HEADER = 'x-datadog-trace-id';
    const DEFAULT_PARENT_ID_HEADER = 'x-datadog-parent-id';
    const DEFAULT_SAMPLING_PRIORITY_HEADER = 'x-datadog-sampling-priority';
    const DEFAULT_ORIGIN_HEADER = 'x-datadog-origin';
    public function inject(SpanContextInterface $spanContext, &$carrier);
    public function extract($carrier);
}
}
namespace DDTrace\Propagators {
use DDTrace\Log\LoggingTrait;
use DDTrace\Propagator;
use DDTrace\Sampling\PrioritySampling;
use DDTrace\Contracts\SpanContext as SpanContextInterface;
use DDTrace\Contracts\Tracer;
use DDTrace\SpanContext;
final class TextMap implements Propagator
{
    use LoggingTrait;
    private $tracer;
    public function __construct(Tracer $tracer)
    {
        $this->tracer = $tracer;
    }
    public function inject(SpanContextInterface $spanContext, &$carrier)
    {
        $carrier[Propagator::DEFAULT_TRACE_ID_HEADER] = $spanContext->getTraceId();
        $carrier[Propagator::DEFAULT_PARENT_ID_HEADER] = $spanContext->getSpanId();
        foreach ($spanContext as $key => $value) {
            $carrier[Propagator::DEFAULT_BAGGAGE_HEADER_PREFIX . $key] = $value;
        }
        $prioritySampling = $this->tracer->getPrioritySampling();
        if (PrioritySampling::UNKNOWN !== $prioritySampling) {
            $carrier[Propagator::DEFAULT_SAMPLING_PRIORITY_HEADER] = $prioritySampling;
        }
        if (!empty($spanContext->origin)) {
            $carrier[Propagator::DEFAULT_ORIGIN_HEADER] = $spanContext->origin;
        }
    }
    public function extract($carrier)
    {
        $traceId = '';
        $spanId = '';
        $baggageItems = [];
        foreach ($carrier as $key => $value) {
            if ($key === Propagator::DEFAULT_TRACE_ID_HEADER) {
                $traceId = $this->extractStringOrFirstArrayElement($value);
            } elseif ($key === Propagator::DEFAULT_PARENT_ID_HEADER) {
                $spanId = $this->extractStringOrFirstArrayElement($value);
            } elseif (strpos($key, Propagator::DEFAULT_BAGGAGE_HEADER_PREFIX) === 0) {
                $baggageItems[substr($key, strlen(Propagator::DEFAULT_BAGGAGE_HEADER_PREFIX))] = $value;
            }
        }
        if (preg_match('/^\d+$/', $traceId) !== 1 || preg_match('/^\d+$/', $spanId) !== 1) {
            return null;
        }
        $spanContext = new SpanContext($traceId, $spanId ?: '', null, $baggageItems, true);
        $this->extractPrioritySampling($spanContext, $carrier);
        $this->extractOrigin($spanContext, $carrier);
        return $spanContext;
    }
    private function extractStringOrFirstArrayElement($value)
    {
        $singleValue = $this->extractScalarOrFirstArrayElement($value);
        if (is_string($singleValue)) {
            return $singleValue;
        }
        return null;
    }
    private function extractScalarOrFirstArrayElement($value)
    {
        if (is_array($value) && count($value) > 0) {
            return $value[0];
        } elseif (is_scalar($value)) {
            return $value;
        }
        return null;
    }
    private function extractPrioritySampling(SpanContextInterface $spanContext, $carrier)
    {
        if (isset($carrier[Propagator::DEFAULT_SAMPLING_PRIORITY_HEADER])) {
            $rawValue = $this->extractScalarOrFirstArrayElement($carrier[Propagator::DEFAULT_SAMPLING_PRIORITY_HEADER]);
            $spanContext->setPropagatedPrioritySampling(PrioritySampling::parse($rawValue));
        }
    }
    private function extractOrigin(SpanContextInterface $spanContext, $carrier)
    {
        if (property_exists($spanContext, 'origin') && isset($carrier[Propagator::DEFAULT_ORIGIN_HEADER])) {
            $spanContext->origin = $this->extractStringOrFirstArrayElement($carrier[Propagator::DEFAULT_ORIGIN_HEADER]);
        }
    }
}
}
namespace DDTrace {
use DDTrace\Contracts\Scope as ScopeInterface;
use DDTrace\Contracts\ScopeManager as ScopeManagerInterface;
use DDTrace\Contracts\Span as SpanInterface;
use DDTrace\Util\ObjectKVStore;
final class ScopeManager implements ScopeManagerInterface
{
    private $scopes = [];
    private $hostRootScopes = [];
    private $rootContext;
    public function __construct($rootContext = null)
    {
        $this->rootContext = $rootContext;
    }
    public function activate(SpanInterface $span, $finishSpanOnClose = self::DEFAULT_FINISH_SPAN_ON_CLOSE)
    {
        $scope = new Scope($this, $span, $finishSpanOnClose);
        if ($span instanceof Span && ObjectKVStore::get($span->internalSpan, 'ddtrace_scope_activated') !== null) {
            ObjectKVStore::put($span->internalSpan, 'ddtrace_scope_activated', true);
        }
        $this->scopes[count($this->scopes)] = $scope;
        if ($span->getContext()->isHostRoot()) {
            $this->hostRootScopes[] = $scope;
        }
        return $scope;
    }
    public function getActive()
    {
        $this->reconcileInternalAndUserland();
        for ($i = count($this->scopes) - 1; $i >= 0; --$i) {
            $scope = $this->scopes[$i];
            $span = $scope->getSpan();
            if (!$span instanceof Span || ObjectKVStore::get($span->internalSpan, 'ddtrace_scope_activated', true)) {
                return $scope;
            }
        }
        return null;
    }
    private function deactivateHostRoot(Scope $scope)
    {
        $i = array_search($scope, $this->hostRootScopes, true);
        if (false !== $i) {
            array_splice($this->hostRootScopes, $i, 1);
        }
    }
    public function deactivate(Scope $scope)
    {
        $i = array_search($scope, $this->scopes, true);
        if (false === $i) {
            return;
        }
        array_splice($this->scopes, $i, 1);
        $span = $scope->getSpan();
        if ($span instanceof Span && ObjectKVStore::get($span->internalSpan, 'ddtrace_scope_activated') !== null) {
            ObjectKVStore::put($span->internalSpan, 'ddtrace_scope_activated', false);
        }
        if ($span->getContext()->isHostRoot()) {
            $this->deactivateHostRoot($scope);
        }
    }
    public function getPrimaryRoot()
    {
        $this->reconcileInternalAndUserland();
        return reset($this->scopes) ?: null;
    }
    public function getTopScope()
    {
        return $this->reconcileInternalAndUserland();
    }
    private function reconcileInternalAndUserland()
    {
        $topScope = null;
        for ($i = count($this->scopes) - 1; $i >= 0; --$i) {
            $topScope = $this->scopes[$i];
            if ($topScope->getSpan()->isFinished()) {
                unset($this->scopes[$i]);
            } else {
                break;
            }
        }
        if (empty($this->scopes)) {
            if ($internalRootSpan = root_span()) {
                $traceId = trace_id();
                if ($this->rootContext) {
                    $parentId = $this->rootContext->spanId;
                } else {
                    $parentId = null;
                }
                $context = new SpanContext($traceId, $internalRootSpan->id, $parentId, []);
                $context->parentContext = $this->rootContext;
                $topScope = $this->scopes[0] = new Scope($this, new Span($internalRootSpan, $context), false);
            } else {
                return null;
            }
        }
        $currentSpanId = $topScope->getSpan()->getSpanId();
        $newScopes = [];
        for ($span = active_span(); $span && $span->id != $currentSpanId; $span = $span->parent) {
            $scope = new Scope($this, new Span($span, new SpanContext(trace_id(), $span->id, $span->parent ? $span->parent->id : null)), true);
            $newScopes[] = $scope;
        }
        foreach (array_reverse($newScopes) as $scope) {
            $scope->getSpan()->getContext()->parentContext = end($this->scopes)->getSpan()->getContext();
            $this->scopes[count($this->scopes)] = $scope;
        }
        return $newScopes ? $newScopes[0] : $topScope;
    }
    public function close()
    {
        foreach ($this->hostRootScopes as $scope) {
            $scope->close();
        }
    }
}
}
namespace DDTrace {
use DDTrace\Contracts\Span as SpanInterface;
use DDTrace\Contracts\SpanContext as SpanContextInterface;
use DDTrace\Contracts\Tracer as TracerInterface;
use DDTrace\Exceptions\UnsupportedFormat;
use DDTrace\Log\LoggingTrait;
use DDTrace\Propagators\Noop as NoopPropagator;
use DDTrace\Propagators\TextMap;
use DDTrace\Transport\Internal;
use DDTrace\Transport\Noop;
use DDTrace\Transport\Noop as NoopTransport;
use DDTrace\Util\ObjectKVStore;
final class Tracer implements TracerInterface
{
    use LoggingTrait;
    private $traces = [];
    private $transport;
    private $propagators;
    private $config = [
        'service_name' => PHP_SAPI,
        'enabled' => true,
        'global_tags' => [],
    ];
    private $scopeManager;
    private $serviceVersion;
    private $environment;
    private $rootContext;
    public function __construct($transport = null, $propagators = null, array $config = [])
    {
        $this->transport = $transport ?: new Internal();
        $textMapPropagator = new TextMap($this);
        $this->propagators = $propagators ?: [Format::TEXT_MAP => $textMapPropagator, Format::HTTP_HEADERS => $textMapPropagator];
        $this->config = array_merge($this->config, $config);
        foreach ($this->config['global_tags'] as $key => $val) {
            add_global_tag($key, $val);
        }
        $this->serviceVersion = \dd_trace_env_config("DD_VERSION");
        $this->environment = \dd_trace_env_config("DD_ENV");
        $context = current_context();
        if (isset($context["distributed_tracing_parent_id"])) {
            $parentId = $context["distributed_tracing_parent_id"];
            $this->rootContext = new SpanContext($context["trace_id"], $parentId, null, [], true);
            if (isset($context["distributed_tracing_origin"])) {
                $this->rootContext->origin = $context["distributed_tracing_origin"];
            }
        }
        $this->scopeManager = new ScopeManager($this->rootContext);
    }
    public function limited()
    {
        return dd_trace_tracer_is_limited();
    }
    public function reset()
    {
        if ($this->config['enabled'] && \ddtrace_config_trace_enabled()) {
            ini_set("datadog.trace.enabled", 0);
            ini_set("datadog.trace.enabled", 1);
        }
        $this->scopeManager = new ScopeManager($this->rootContext);
    }
    public static function noop()
    {
        return new self(new NoopTransport(), [Format::BINARY => new NoopPropagator(), Format::TEXT_MAP => new NoopPropagator(), Format::HTTP_HEADERS => new NoopPropagator()], ['enabled' => false]);
    }
    public function startSpan($operationName, $options = [])
    {
        if (!$this->config['enabled'] || !\ddtrace_config_trace_enabled()) {
            return NoopSpan::create();
        }
        if (!$options instanceof StartSpanOptions) {
            $options = StartSpanOptions::create($options);
        }
        $reference = $this->findParent($options->getReferences());
        $roundedStartTime = $options->getStartTime() ? ($options->getStartTime() + 0.2) / 1000000 : 0;
        if ($reference === null) {
            $context = SpanContext::createAsRoot([], $roundedStartTime);
        } else {
            $context = SpanContext::createAsChild($reference->getContext(), $roundedStartTime);
        }
        $resource = array_key_exists('resource', $this->config) ? (string) $this->config['resource'] : null;
        $service = $this->config['service_name'];
        $internalSpan = active_span();
        $internalSpan->name = (string) $operationName;
        $internalSpan->service = $service;
        $internalSpan->resource = $resource;
        if (!isset($internalSpan->metrics)) {
            $internalSpan->metrics = [];
        }
        if (!isset($internalSpan->meta)) {
            $internalSpan->meta = [];
        }
        $span = new Span($internalSpan, $context);
        foreach ($options->getTags() as $key => $val) {
            $span->setTag($key, $val);
        }
        ObjectKVStore::put($internalSpan, 'ddtrace_scope_activated', false);
        return $span;
    }
    public function startRootSpan($operationName, $options = [])
    {
        $rootScope = $this->startActiveSpan($operationName, $options);
        $this->setPrioritySamplingFromSpan($rootScope->getSpan());
        return $rootScope;
    }
    public function getRootScope()
    {
        return $this->scopeManager->getPrimaryRoot();
    }
    public function startActiveSpan($operationName, $options = [])
    {
        if (!$options instanceof StartSpanOptions) {
            $options = StartSpanOptions::create($options);
        }
        if (!root_span() && !$options->getReferences() && $this->rootContext) {
            $options = $options->withParent($this->rootContext);
        }
        $parentService = null;
        if (($activeScope = $this->scopeManager->getTopScope()) !== null) {
            $activeSpan = $activeScope->getSpan();
            $tags = $options->getTags();
            if (!array_key_exists(Tag::SERVICE_NAME, $tags)) {
                $parentService = $activeSpan->getService();
            }
            $options = $options->withParent($activeSpan);
        }
        $span = $this->startSpan($operationName, $options);
        if ($parentService !== null) {
            $span->setTag(Tag::SERVICE_NAME, $parentService);
        }
        $shouldFinish = $options->shouldFinishSpanOnClose() && ($span->getParentId() != 0 || !\dd_trace_env_config('DD_TRACE_GENERATE_ROOT_SPAN'));
        return $this->scopeManager->activate($span, $shouldFinish);
    }
    private function findParent(array $references)
    {
        foreach ($references as $reference) {
            if ($reference->isType(Reference::CHILD_OF)) {
                return $reference;
            }
        }
        return null;
    }
    public function inject(SpanContextInterface $spanContext, $format, &$carrier)
    {
        if (!array_key_exists($format, $this->propagators)) {
            throw UnsupportedFormat::forFormat($format);
        }
        $this->propagators[$format]->inject($spanContext, $carrier);
    }
    public function extract($format, $carrier)
    {
        if (array_key_exists($format, $this->propagators)) {
            return $this->propagators[$format]->extract($carrier);
        }
        throw UnsupportedFormat::forFormat($format);
    }
    public function flush()
    {
        if (!$this->config['enabled']) {
            return;
        }
        if ('cli' !== PHP_SAPI && \dd_trace_env_config("DD_TRACE_URL_AS_RESOURCE_NAMES_ENABLED") && $rootSpan = $this->getSafeRootSpan()) {
            $this->addUrlAsResourceNameToSpan($rootSpan);
        }
        if (self::isLogDebugActive()) {
            self::logDebug('Flushing {count} traces, {spanCount} spans', ['count' => $this->getTracesCount(), 'spanCount' => dd_trace_closed_spans_count()]);
        }
        $this->transport->send($this);
        $this->scopeManager->close();
    }
    public function getScopeManager()
    {
        return $this->scopeManager;
    }
    public function getActiveSpan()
    {
        if (null !== $activeScope = $this->scopeManager->getActive()) {
            return $activeScope->getSpan();
        }
        return null;
    }
    public function getTracesAsArray()
    {
        $trace = \dd_trace_serialize_closed_spans();
        return $trace ? [$trace] : $trace;
    }
    public function addUrlAsResourceNameToSpan(Contracts\Span $span)
    {
        if (null !== $span->getResource()) {
            return;
        }
        if (!isset($_SERVER['REQUEST_METHOD'])) {
            return;
        }
        $resourceName = $_SERVER['REQUEST_METHOD'];
        if (isset($_SERVER['REQUEST_URI'])) {
            $resourceName .= ' ' . \DDTrace\Util\Normalizer::uriNormalizeIncomingPath($_SERVER['REQUEST_URI']);
        }
        $span->setTag(Tag::RESOURCE_NAME, $resourceName, true);
    }
    private function setPrioritySamplingFromSpan(SpanInterface $span)
    {
        if (!$span->getContext()->isHostRoot()) {
            return;
        }
        $prioritySampling = $span->getContext()->getPropagatedPrioritySampling();
        if (null !== $prioritySampling) {
            set_priority_sampling($prioritySampling);
        }
    }
    public function setPrioritySampling($prioritySampling)
    {
        set_priority_sampling($prioritySampling);
        set_priority_sampling($prioritySampling, true);
    }
    public function getPrioritySampling()
    {
        return get_priority_sampling();
    }
    public function getSafeRootSpan()
    {
        $rootScope = $this->getRootScope();
        if (empty($rootScope)) {
            return null;
        }
        return $rootScope->getSpan();
    }
    public static function version()
    {
        return \phpversion("ddtrace");
    }
    public function getTracesCount()
    {
        return count($this->traces);
    }
}
}
